//
//  ViewController.swift
//  Face
//

import UIKit

/**
 
 ViewController for showing and manipulating
 a FaceExpression model
 
 */
class ViewController: UIViewController {
  
  @IBOutlet weak var faceView: FaceView! {
    didSet {
      updateUI()
    }
  }
  
  private var expression = FacialExpression(eyes: .open, mouth: .smile) {
    didSet {
      updateUI()
      saveToUserDefaults()
    }
  }
  
  /// maps Mouth enum values to appropriate mouthCurvatures
  private let mouthCurvatures : [FacialExpression.Mouth : CGFloat] = [
    .frown : -1,
    .smirk : -0.5,
    .neutral : 0,
    .grin : 0.5,
    .smile : 1
  ]
  
  /// update the faceView to reflect the model value
  private func updateUI() {
    switch (expression.eyes) {
    case .closed:
      faceView.eyesOpen = false
    case .open:
      faceView.eyesOpen = true
    case .squinting:
      faceView.eyesOpen = false
    }
    faceView.mouthCurvature = mouthCurvatures[expression.mouth] ?? 0.0
  }
  
  // Actions for the gestures that change the model.
  
  @IBAction func toggleEyes(_ sender: UITapGestureRecognizer) {
    if sender.state == .ended {
      let eyes : FacialExpression.Eyes = (expression.eyes == .open) ? .closed : .open
      expression = FacialExpression(eyes: eyes, mouth: expression.mouth)
    }
  }
  
  @IBAction func makeHappier(_ sender: UISwipeGestureRecognizer) {
    if sender.state == .ended {
      expression = expression.happier()
    }
  }
  
  @IBAction func makeSadder(_ sender: UISwipeGestureRecognizer) {
    if sender.state == .ended {
      expression = expression.sadder()
    }
  }
  
  
  func saveToUserDefaults() {
    let defaults = UserDefaults.standard
    defaults.set(expression.eyes.rawValue, forKey: "eyes")
    defaults.set(expression.mouth.rawValue, forKey: "mouth")
  }
  
  func loadFromUserDefaults() -> FacialExpression {
    let defaults = UserDefaults.standard
    let eyes = FacialExpression.Eyes(rawValue: defaults.integer(forKey: "eyes")) ?? .closed
    let mouth = FacialExpression.Mouth(rawValue: defaults.integer(forKey: "mouth")) ?? .neutral
    return FacialExpression(eyes: eyes, mouth: mouth)
  }
  
  override func viewDidLoad() {
    super.viewDidLoad()
    expression = loadFromUserDefaults()
  }
  
  
}

