//
//  FaceView.swift
//
//  Modified by Steve for cs 326.
//
//  Created by CS193p Instructor.
//  Copyright © 2015-17 Stanford University. All rights reserved.
//
//

import UIKit

/**
 
 A custom view for drawing smiling (or frowning) faces.  Based on
 example
 
 */
@IBDesignable
public class FaceView: UIView {
  
  // Public properties
  
  @IBInspectable
  public var scale : CGFloat = 0.9 {
    didSet {
      setNeedsDisplay()
    }
  }
  
  @IBInspectable
  public var eyesOpen : Bool = true  {
    didSet {
      setNeedsDisplay()
    }
  }
  
  /// How "smiley" the face is.  1 is full smile. -1 is full frown.
  /// Values outside of that range will be clamped to within the range.
  @IBInspectable
  public var mouthCurvature: CGFloat = 1 {
    didSet {
      mouthCurvature = max(min(mouthCurvature, 1), -1)
      setNeedsDisplay() 
    }
  }
  
  @IBInspectable
  public var color : UIColor = UIColor.blue  {
    didSet {
      setNeedsDisplay()
    }
  }
  
  @IBInspectable
  public var lineThickness : CGFloat = 5.0  {
    didSet {
      setNeedsDisplay()
    }
  }
  
  // Private implementation
  
  
  /// Action to adjust scale for a pinch gesture.  It is
  /// in FaceView instead of the ViewController because it
  /// really just affects the view and does not the model.
  @IBAction func changeScale(_ pinchRecognizer : UIPinchGestureRecognizer) {
    switch pinchRecognizer.state {
    case .changed, .ended:
      scale *= pinchRecognizer.scale
      pinchRecognizer.scale = 1
    default:
      break
    }
  }

  
  private var headCenter : CGPoint {
    return CGPoint(x: bounds.midX, y: bounds.midY)
  }
  
  private var headRadius : CGFloat {
    let boundsSize = bounds.size
    return min(boundsSize.width, boundsSize.height) / 2 * scale
  }
  
  private struct Ratios {
    static let headRadiusToEyeOffset: CGFloat = 3
    static let headRadiusToEyeRadius: CGFloat = 10
    static let headRadiusToMouthWidth: CGFloat = 1
    static let headRadiusToMouthHeight: CGFloat = 3
    static let headRadiusToMouthOffset: CGFloat = 3
  }
  
  private enum Eye {
    case left
    case right
  }
  
  private func headPath() -> UIBezierPath {
    let head = UIBezierPath(arcCenter: headCenter,
                            radius: headRadius,
                            startAngle: 0,
                            endAngle: CGFloat.pi*2,
                            clockwise: true)
    head.lineWidth = lineThickness
    return head
  }
  
  private func eyePath(_ eye: Eye) -> UIBezierPath {
    let eyeOffset = headRadius / Ratios.headRadiusToEyeOffset
    let eyeCenter = CGPoint(x: headCenter.x + ((eye == .left) ? -1 : 1) * eyeOffset,
                            y: headCenter.y - eyeOffset)
    let eyeRadius = headRadius / Ratios.headRadiusToEyeRadius
    var path : UIBezierPath
    if eyesOpen {
      path = UIBezierPath(ovalIn: CGRect(x: eyeCenter.x - eyeRadius,
                                         y: eyeCenter.y - eyeRadius,
                                         width: 2 * eyeRadius,
                                         height: 2 * eyeRadius))
    } else {
      path = UIBezierPath()
      path.move(to: CGPoint(x: eyeCenter.x - eyeRadius, y: eyeCenter.y))
      path.addLine(to: CGPoint(x: eyeCenter.x + eyeRadius, y: eyeCenter.y))
    }
    path.lineWidth = lineThickness
    return path
  }
  
  private func mouthPath() -> UIBezierPath {
    let mouthWidth = headRadius / Ratios.headRadiusToMouthWidth
    let mouthHeight = headRadius / Ratios.headRadiusToMouthHeight
    let mouthOffset = headRadius / Ratios.headRadiusToMouthOffset
    let mouthRect = CGRect(x: headCenter.x - mouthWidth / 2,
                           y: headCenter.y + mouthOffset,
                           width: mouthWidth,
                           height: mouthHeight)
    
    let smileOffset = CGFloat(mouthCurvature) * mouthHeight
    
    let start = CGPoint(x: mouthRect.minX, y: mouthRect.midY)
    let end = CGPoint(x: mouthRect.maxX, y: mouthRect.midY)
    let controlPoint1 = CGPoint(x: start.x + mouthWidth / 3, y: start.y + smileOffset)
    let controlPoint2 = CGPoint(x: end.x - mouthWidth / 3, y: start.y + smileOffset)
    
    let path = UIBezierPath()
    path.move(to: start)
    path.addCurve(to: end, controlPoint1: controlPoint1, controlPoint2: controlPoint2)
    path.lineWidth = lineThickness
    
    return path
  }
  
  
  /**
   
   Called when the view should be refreshed.  You may ignore the
   rect parameter, which describes which region of the view should
   be redrawn.  We'll always just redraw the whole view.
   
   */
  override public func draw(_ rect: CGRect) {
    color.setStroke()
    headPath().stroke()
    eyePath(.left).stroke()
    eyePath(.right).stroke()
    mouthPath().stroke()
  }
  
  
}
