// A simple clock for use in timing things.
// (c) 1998 McGraw-Hill
package structure;

/**
 * A simple object for measuring time.
 * This Clock allows one to measure time between events.  Events may be
 * from several milliseconds to several seconds apart.  When measuring
 * events that take considerable time (> 1/10th of a second) it is important
 * to counteract the effect of scheduling of other system processes.
 * <p>
 * A Clock has the following operations:
 *  <ul>
 *  <li> reset - clear the clock and reset accumulated time to zero
 *  <li> start - start the clock and accumulate time
 *  <li> stop  - stop the clock (and therefore stop accumulating time)
 *  <li> read  - read a stopped clock to get accumulated time in milliseconds.
 * </ul>
 *
 * typical use:
 * <pre>
 *    Clock timer = new Clock();
 *    timer.start();
 *    for (i = 0; i < 100000; i++);
 *    timer.stop();
 *    System.out.println("Time to count to 100000: "+timer.read()+" seconds.");
 * </pre>
 *
 * @version $Id: Clock.java,v 2.1 1999/07/11 02:51:11 bailey Exp bailey $
 * @author duane a. bailey
 */
public class Clock {
    // we use a native-code library for structures
    /**
     * An indication of whether or not the clock is running
     */
    private boolean running;    // is the clock on?
    /**
     * The millisecond that the clock started.
     */
    private long strt;  // starting millisecond count
    /**
     * The total number of milliseconds elapsed.
     */
    private long accum; // total milliseconds

    /**
     * Constructs a stopwatch for timing events to the milliseconds.
     * <p>
     * <dl>
     * <dt><b>Postcondition:</b><dd> returns a stopped clock
     * </dl>
     */
    public Clock()
    // post: returns a stopped clock
    {
        running = false;
        strt = 0;
        accum = 0;
    }

    /**
     * Start the clock running.
     * <p>
     * <dl>
     * <dt><b>Postcondition:</b><dd> clock is stopped
     * <dt><b>Precondition:</b><dd> starts clock, begins measuring possibly accumulated time
     * </dl>
     */
    public void start()
    // post: clock is stopped
    // pre: starts clock, begins measuring possibly accumulated time
    {
        running = true;
        strt = System.currentTimeMillis();
    }

    /**
     * Stop the clock.  Time does not accumulate.
     * <p>
     * <dl>
     * <dt><b>Precondition:</b><dd> clock is running
     * <dt><b>Postcondition:</b><dd> stops clock, and accumulates time
     * </dl>
     */
    public void stop()
    // pre: clock is running
    // post: stops clock, and accumulates time
    {
        running = false;
        accum += (System.currentTimeMillis()-strt);
    }

    /**
     * Read the value on the stop watch.
     * <p>
     * <dl>
     * <dt><b>Precondition:</b><dd> clock is stopped.
     * <dt><b>Postcondition:</b><dd> returns the accumulated time on the clock
     * </dl>
     * 
     * @return A double representing the number of seconds elapsed.
     */
    public double read()
    // pre: clock is stopped.
    // post: returns the accumulated time on the clock
    {
        return (double)accum/(double)1000.0;
    }

    /**
     * Resets the time on the clock to zero.
     * <p>
     * <dl>
     * <dt><b>Postcondition:</b><dd> stops running clock and clears the accumulated time.
     * </dl>
     */
    public void reset()
    // post: stops running clock and clears the accumulated time.
    {
        running = false;
        accum = 0;
    }

    /**
     * Generates string representation of clock.
     * <p>
     * <dl>
     * <dt><b>Postcondition:</b><dd> returns a string representation of the clock
     * </dl>
     * 
     * @return A string representing this clock.
     */
    public String toString()
    // post: returns a string representation of the clock
    {
        return "<Clock: "+read()+" seconds>";
    }
}
