// Implements a simple line class.
// (c) 1997 duane a. bailey
package element;
import java.awt.*;

/**
 * Copyright (c) 1997 McGraw-Hill
 * All Rights Reserved.
 * <p>
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies. Please refer to
 * the file <a href="http://www.cs.williams.edu">"copyright.html"</a> for further important copyright
 * and licensing information.
 * <p>
 * MCGRAW-HILL MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NON-INFRINGEMENT. MCGRAW-HILL SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 * @version $Id: Text.java,v 2.2 1999/08/05 16:18:01 bailey Exp bailey $
 * @author duane a. bailey
 */
public class Text implements Drawable
{
    private int left, bottom;		// ordered
    private int width, height;
    private Font font;
    private FontMetrics fm;
    private String string;
    private static Graphics context;
    private static Font defaultFont;
    private static FontMetrics dfm;
    
    /**
     * @param s 
     */
    public Text(String s)
    // pre: s is a valid string
    // post: construct a text using s, with lower left at origin
    {
	this(s,0,0);
    }

    /**
     * @param c 
     */
    public Text(char c)
    // post: construct a text using c, with lower left at origin
    {
	this(new Character(c).toString(),0,0);
    }

    /**
     * @param i 
     */
    public Text(long i)
    // post: construct a text using i, with lower left at origin
    {
	this(Long.toString(i),0,0);
    }

    /**
     * @param s 
     */
    public Text(Object s)
    // post: s is a not null object
    // post: construct a Text representation of s at origin
    {
	this(s,0,0);
    }

    /**
     * @param s 
     * @param left 
     * @param bottom 
     */
    public Text(Object s, int left, int bottom)
    // post: s is a not null object
    // post: construct a text using s.toString;
    //       with lower left at (left,bottom)
    {
	this(s.toString(),left,bottom);
    }

    /**
     * @param s 
     * @param left 
     * @param bottom 
     */
    public Text(String s, int left, int bottom)
    // post: s is a not null String
    // post: construct a text using s
    //       with lower left at (left,bottom)
    {
	if (defaultFont == null)
	{
	    defaultFont = new Font("Courier",Font.PLAIN,12);
	    dfm = null;
	}
	string = new String(s);
	this.left = left;
	this.bottom = bottom;
	font = defaultFont;
	fm = null;
	getDimensions();
    }

    public Text(Object s, Pt p)
    // pre: s is not null, p is not null
    // post: construct rep of s with lower left at p
    {
        this(s,p.x(),p.y());
    }

    /**
     * @param that 
     */
    public Text(Text that)
    // pre: that is not null
    // post: this is a copy of that
    {
	string = that.string;
	left = that.left;
	bottom = that.bottom;
	font = that.font;
	fm = that.fm;
	width = that.width;
	height = that.height;
    }

    private void getDimensions()
    {
	if (height != 0) return;
	if (fm == null) {
	    if (dfm == null)
	    {
		if (context == null) {
		    if (DrawingWindow.exampleContext == null) {
			return;
		    }
		    context = DrawingWindow.exampleContext;
		    dfm = context.getFontMetrics(defaultFont);
		}
	    }
	    if (font == defaultFont) {
		fm = dfm;
	    } else {
		fm = context.getFontMetrics(font);
	    }
	}
	width = fm.stringWidth(string);
	height = fm.getAscent() + fm.getDescent();
    }

    /**
     * @return 
     */
    public int left()
    // post: returns the left coordinate of the text
    {
	return left;
    }

    /**
     * @return 
     */
    public int right()
    // post: returns the right coordinate of the text
    {
	return left + width();
    }

    /**
     * @return 
     */
    public int top()
    // post: returns the top coordinate of the text
    {
	return bottom - height();
    }

    /**
     * @return 
     */
    public int bottom()
    // post: returns the bottom coordinate of the text
    {
	return bottom;
    }
    
    /**
     * @param x 
     */
    public void left(int x)
    // post: sets the left coordinate of the text,
    //       relocating text
    {
	left = x;
    }

    /**
     * @param x 
     */
    public void right(int x)
    // post: sets the right coordinate of the text,
    //      relocating text
    {
	left = x-width();
    }

    /**
     * @param y 
     */
    public void top(int y)
    // post: sets the top coordinate of the text, relocating text
    {
	bottom = y+height();
    }

    /**
     * @param y 
     */
    public void bottom(int y)
    // post: sets the bottom coordinate of the text,
    //      relocating text
    {
	bottom = y;
    }

    /**
     * @return 
     */
    public int width()
    // post: returns the width of the text
    {
	getDimensions();
	return width;
    }

    /**
     * @return 
     */
    public int height()
    // post: returns the height of the text
    {
	getDimensions();
	return height;
    }

    /**
     * @return 
     */
    public Pt center()
    // post: returns the center of the text
    {
	return new Pt(left + width()/2, bottom-height()/2);
    }

    /**
     * @param p 
     */
    public void center(Pt p)
    // pre: p is not null
    // post: relocates the center of the text at p
    {
	Pt q = center();
	left(left()+(p.x()-q.x()));
	top(top()+(p.y()-q.y()));
    }

    /**
     * @param p 
     * @return 
     */
    public boolean contains(Pt p)
    // pre: returns true if the text's bounding rect contains p
    {
	return Line.within(p.x(),left,width()) &&
	       Line.within(p.y(),top(),height());
    }

    /**
     * @param d 
     */
    public void fillOn(DrawingWindow d)
    // post: draws the text on the drawing window
    {
	d.drawText(this);
    }

    /**
     * @param d 
     */
    public void clearOn(DrawingWindow d)
    // post: erases the text from the drawing window
    {
	d.clearText(this);
    }

    /**
     * @param d 
     */
    public void drawOn(DrawingWindow d)
    // post: draws the text on the drawing window
    {
	d.drawText(this);
    }

    /**
     * post: returns suitable hash code<br>
     * 
     * @return 
     */
    public int hashCode()
    // post: returns suitable hash code
    {
	return string.hashCode();
    }

    /**
     * post: returns true if two texts are equal<br>
     * 
     * @param other 
     * @return 
     */
    public boolean equals(Object other)
    // pre: other is not null Text
    // post: returns true if two Texts are equal
    {
	Text that = (Text)other;
	return (this.string.equals(that.string)) &&
	       (this.left == that.left) &&
	       (this.bottom == that.bottom);
    }

    /**
     * post: returns a distinct copy of the Object<br>
     * 
     * @return 
     */
    public Object clone()
    // post: returns a distinct copy of the Object
    {
	return new Text(this);
    }

    /**
     * @return 
     */
    public String string()
    {
	return string;
    }

    /**
     * @return 
     */
    public Font font()
    {
	return font;
    }

    /**
     * post: returns a string representation of Object<br>
     * 
     * @return 
     */
    public String toString()
    // post: returns a string representation of Object
    {
	return "<Text: \""+string+"\" left="+left+" bottom="+bottom+
	       " width="+width()+" height="+height()+">";
    }
}

