// Implements a simple rounded rectangle class.
// (c) 1997 duane a. bailey
package element;

/**
 * Copyright (c) 1997 McGraw-Hill
 * All Rights Reserved.
 * <p>
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this
 * copyright notice appears in all copies. Please refer to
 * the file <a href="http://www.cs.williams.edu">"copyright.html"</a> for further important copyright
 * and licensing information.
 * <p>
 * MCGRAW-HILL MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NON-INFRINGEMENT. MCGRAW-HILL SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 * @version $Id: RoundRect.java,v 2.2 1999/08/05 16:18:01 bailey Exp bailey $
 * @author duane a. bailey
 */
public class RoundRect extends Rect
{
    protected int cornerWidth, cornerHeight;

    /**
     * post: constructs a trivial rectangle at origin<br>
     * 
     */
    public RoundRect()
    // post: constructs a trivial rectangle at origin
    {
	this(0,0,0,0,0,0);
    }

    /**
     * @param p 
     * @param q 
     */
    public RoundRect(Pt p, Pt q)
    // pre: p and q are valid points
    // post: constructs a RoundRect bounded by p and q
    {
	super(p,q);
	cornerWidth = width()/2;
	cornerHeight = height()/2;
    }

    /**
     * post: constructs a rounded rectangle<br>
     * 
     * @param r 
     * @param cw 
     * @param ch 
     */
    public RoundRect(Rect r, int cw, int ch)
    // post: constructs a rounded rectangle
    {
	this(r.left(),r.top(),r.width(),r.height(),cw,ch);
    }

    /**
     * pre:  w >= 0, h >= 0<br>
     * post: constructs rectangle with top left at (x,y)<br>
     *       width w, height h<br>
     * 
     * @param x 
     * @param y 
     * @param w 
     * @param h 
     * @param cw 
     * @param ch 
     */
    public RoundRect(int x, int y, int w, int h, int cw, int ch)
    // pre:  w >= 0, h >= 0
    // post: constructs rectangle with top left at (x,y),
    //       width w, height h
    {
	super(x,y,w,h);
	cornerWidth = Math.max(cw,0);
	cornerHeight = Math.max(ch,0);
    }

    /**
     * pre:  w >= 0, h >= 0<br>
     * post: constructs rectangle with top left at (x,y)<br>
     *       width w, height h<br>
     * 
     * @param x 
     * @param y 
     * @param w 
     * @param h 
     */
    public RoundRect(int x, int y, int w, int h)
    // pre:  w >= 0, h >= 0
    // post: constructs rectangle with top left at (x,y),
    //       width w, height h
    {
	super(x,y,w,h);
	cornerWidth = width()/2;
	cornerHeight = height()/2;
    }

    /**
     * @param r 
     */
    public RoundRect(RoundRect r)
    // pre: r is not null
    // post: constructs a rounded rectangle like r
    {
	super((Rect)r);
	cornerWidth = r.cornerWidth;
	cornerHeight = r.cornerHeight;
    }

    /**
     * @return 
     */
    public int cornerWidth()
    // post: returns the width of the corner oval
    {
	return cornerWidth;
    }

    /**
     * @return 
     */
    public int cornerHeight()
    // post: returns the height of the corner oval
    {
	return cornerHeight;
    }

    /**
     * @param cw 
     */
    public void cornerWidth(int cw)
    // pre: cw >= 0
    // post: sets the width of the corner oval    
    {
	cornerWidth = Math.max(cw,0);
    }

    /**
     * @param ch 
     */
    public void cornerHeight(int ch)
    // pre: ch > 0
    // post: sets the height of the corner oval
    {
	cornerHeight = Math.max(ch,0);
    }

    /**
     * @param p 
     * @return 
     */
    public boolean contains(Pt p)
    // pre: p is not null
    // post: returns true iff p is within the rounded rect
    {
	Rect inner = new Rect(this);
	inner.extend(-cornerWidth/2,0);
	if (inner.contains(p)) return true;
	inner.extend(cornerWidth/2,-cornerHeight/2);
	if (inner.contains(p)) return true;
	inner = new Oval(0,0,cornerWidth,cornerHeight);
	inner.left(left());
	inner.top(top());
	if (inner.contains(p)) return true;
	inner.right(right());
	if (inner.contains(p)) return true;
	inner.bottom(bottom());
	if (inner.contains(p)) return true;
	inner.left(left());
	return inner.contains(p);
    }

    /**
     * @param d 
     */
    public void fillOn(DrawingWindow d)
    // pre: d is not null
    // post: draws this filled rounded rectangle on d
    {
	d.fillRoundRect(left,top,width,height,cornerWidth,cornerHeight);
    }

    /**
     * @param d 
     */
    public void clearOn(DrawingWindow d)
    // pre: d is not null
    // post: erases this rounded rectangle on d
    {
	d.clearRoundRect(left,top,width,height,cornerWidth,cornerHeight);
    }

    /**
     * @param d 
     */
    public void drawOn(DrawingWindow d)
    // pre: d is not null
    // post: draws this rounded rectangle on d
    {
	d.drawRoundRect(left,top,width,height,cornerWidth,cornerHeight);
    }

    /**
     * post: returns suitable hash code<br>
     * 
     * @return 
     */
    public int hashCode()
    // post: returns suitable hash code
    {
	return left+top+width+height+cornerWidth+cornerHeight;
    }

    /**
     * post: returns true if two round rects are equal<br>
     * 
     * @param other 
     * @return 
     */
    public boolean equals(Object other)
    // pre: other is not null RoundRect
    // post: returns true if two RoundRects are equal
    {
	RoundRect that = (RoundRect)other;
	return (this.left == that.left) &&
	       (this.top == that.top) &&
	       (this.width == that.width) &&
	       (this.height == that.height) &&
	       (this.cornerWidth == that.cornerWidth) &&
	       (this.cornerHeight == that.cornerHeight);
    }

    /**
     * post: returns a distinct copy of the Object<br>
     * 
     * @return 
     */
    public Object clone()
    // post: returns a distinct copy of the Object
    {
	return new RoundRect(this);
    }

    /**
     * post: returns a string representation of Object<br>
     * 
     * @return 
     */
    public String toString()
    // post: returns a string representation of Object
    {
	return "<RoundRect: left="+left+" top="+top+" width="+width+" height="+height+" corner width="+cornerWidth+" corner height="+cornerHeight+">";
    }
}
