// Implements a simple rectangle class.
// (c) 1997, 1998 duane a. bailey
package element;
import java.awt.Rectangle;

/**
 * An implementation of a simple graphics rectangle.
 *
 * @version $Id: Rect.java,v 2.2 1999/08/05 16:18:01 bailey Exp bailey $
 * @author duane a. bailey
 */
public class Rect implements Drawable
{
    /**
     * The left side of the rectangle.
     */
    protected int left;
    /**
     * The top side of the rectangle.
     */
    protected int top;
    /**
     * The width of the rectangle.
     */
    protected int width;
    /**
     * The height of the rectangle.
     */
    protected int height;

    /**
     * Constructs a trivial rectangle, about the origin.
     */
    public Rect()
    // post: constructs a trivial rectangle at origin
    {
	this(0,0,0,0);
    }

    /**
     * Constructs the rectangle determined by two points, p1 and p2.
     * 
     * @param p1 one corner of the rectangle
     * @param p2 the opposite corner of the rectangle
     */
    public Rect(Pt p1, Pt p2)
    // post: constructs a rectangle between p1 and p2
    {
	int x1 = p1.x();
	int x2 = p2.x();
	int y1 = p1.y();
	int y2 = p2.y();

	// construct the canonical form of the rectangle
	left = Math.min(x1,x2);
	top = Math.min(y1,y2);
	width = Math.abs(x1-x2);
	height = Math.abs(y1-y2);
    }

    /**
     * Construct a rectangle from the bounding box of another drawable
     * object.
     * @param o an object to be used as the basis for the constructed rect.
     */
    public Rect(Drawable o)
    // post: constructs a rectangle, based on another drawable object
    {
	this.left = o.left();
	this.top = o.top();
	this.width = o.width();
	this.height = o.height();
    }

    /**
     * Construct a square centered about (cx,cy), with radius r.
     *
     * @param cx the x coordinate of the center
     * @param cy the y coordinate of the center
     * @param r the radius of the rectangle
     */
    public Rect(int cx, int cy, int r)
    // pre: radius >= 0
    // post: constructs a radius r square centered about (cx,cy)
    {
	this(cx-r, cy-r, 2*r, 2*r);
    }

    /**
     * Construct a square with radius r, centered about a point p.
     *
     * @param p the center of the square
     * @param r the radius of the square
     */
    public Rect(Pt p, int r)
    // pre: radius >= 0
    // post: constructs a radius r square centered about (cx,cy)
    {
	this(p.x(), p.y(), r);
    }

    /**
     * Construct a rectangle with upper left (x,y) and dimensions (w,h).
     * 
     * @param x the left of the rectangle
     * @param y the top of the rectangle
     * @param w the width of the rectangle
     * @param h the height of the rectangle
     */
    public Rect(int x, int y, int w, int h)
    // pre:  w >= 0, h >= 0
    // post: constructs a rectangle with upper left (x,y),
    //       width w, height h
    {
	left = x;
	top = y;
	width = (w >= 0)? w : 0;
	height = (h >= 0)? h : 0;
    }

    /**
     * Returns true if p is contained within rectangle
     * @param p the point to be considered
     * @return true iff p is contained within the rectangle
     */
    public boolean contains(Pt p)
    // pre: p is a valid point
    // post: true iff p is within the rectangle
    {
	return Line.within(p.x(),left,width) &&
	       Line.within(p.y(),top,height);
    }

    /**
     * Return the left coordinate of the rectangle.
     * 
     * @return the left coordinate of the rectangle
     */
    public int left()
    // post: returns left coordinate of the rectangle
    {
	return left;
    }

    /**
     * Returns the top coordinate of the rectangle
     * 
     * @return the top coordinate of the rectangle
     */
    public int top()
    // post: returns top coordinate of the rectangle
    {
	return top;
    }

    /**
     * Returns right coordinate of the rectangle
     * 
     * @return the right coordinate of the rectangle
     */
    public int right()
    // post: returns right coordinate of the rectangle
    {
	return left+width;
    }

    /**
     * Returns the bottom coordinate of the rectangle
     * 
     * @return the bottom coordinate of the rectangle
     */
    public int bottom()
    // post: returns the bottom coordinate of the rectangle
    {
	return top+height;
    }

    /**
     * Returns the width of the rectangle.
     * 
     * @return the width of the rectangle
     */
    public int width()
    // post: returns the width of the rectangle
    {
	return width;
    }

    /**
     * Returns the height of the rectangle.
     * 
     * @return the height of the rectangle
     */
    public int height()
    // post: returns the height of the rectangle
    {
	return height;
    }

    /**
     * Sets the width of the rectangle.  Center and height remain unchanged.
     * 
     * @param w the desired width.
     */
    public void width(int w)
    // post: sets width of rectangle, center and height unchanged
    {
	w = Math.max(w,0);
	left += (width-w)/2;
	width = w;
    }

    /**
     * Sets the height of the rectangle.  Center and width remain unchanged.
     * 
     * @param h the desired height.
     */
    public void height(int h)
    // post: sets height of the Rect; center and width unchanged
    {
	h = Math.max(h,0);
	top += (height-h)/2;
	height = h;
    }

    /**
     * Sets the left side of the rectangle.  Dimensions remain unchanged.
     * 
     * @param x the desired left side.
     */
    public void left(int x)
    // post: sets left to x; dimensions remain unchanged
    {
	left = x;
    }

    /**
     * Sets the top side of the rectangle.  Dimensions remain unchanged.
     * 
     * @param y the new top coordinate.
     */
    public void top(int y)
    // post: sets top to y; dimensions remain unchanged
    {
	top = y;
    }

    /**
     * Sets the bottom side of rectangle.  Dimensions remain unchanged.
     * 
     * @param y the new bottom coordinate.
     */
    public void bottom(int y)
    // post: sets bottom to y; dimensions remain unchanged
    {
	top = y-height;
    }

    /**
     * Sets the right coordinate of rectangle.  Dimensions remain unchanged.
     * 
     * @param x the new right coordinate.
     */
    public void right(int x)
    // post: sets the left coordinate; dimensions unchanged
    {
	left = x-width;
    }

    /**
     * Returns the center point of the rectangle.
     * 
     * @return the center point of the rectangle
     */
    public Pt center()
    // post: returns center point of rectangle
    {
	return new Pt(left+width/2,top+height/2);
    }

    /**
     * Sets the center point of the rectangle.  The dimensions remain
     * unchanged.
     * @param p the new center point of the rectangle
     */
    public void center(Pt p)
    // post: sets center of rect to p; dimensions remain unchanged
    {
	left(p.x()-width/2);
	top(p.y()-height/2);
    }

    /**
     * Moves the rectangle in the direction (dx,dy).
     * 
     * @param dx the horizontal change in position
     * @param dy the vertical change in position
     */
    public void move(int dx, int dy)
    // post: moves rectangle to left by dx and down by dy
    {
	left += dx;
	top += dy;
    }

    /**
     * Moves the top-left coordinate of the rectangle to (left,top).
     * The dimensions remain unchanged.
     *
     * @param left the new left coordinate 
     * @param top the new top coordinate
     */
    public void moveTo(int left, int top)
    // post: moves left top of rectangle to (left,top);
    //       dimensions are unchanged
    {
	this.left = left;
	this.top = top;
    }

    /**
     * Moves the top-left coordinate to point p
     * 
     * @param p the new top left position
     */
    public void moveTo(Pt p)
    // post: moves left top of rectangle to p
    {
	moveTo(p.x(), p.y());
    }

    /**
     * Moves the sides of the rectangle outward by dx and dy.
     * 
     * @param dx the change in horizontal coordinates - positive is outward
     * @param dy the change in vertical coordinates - positive is outward
     */
    public void extend(int dx, int dy)
    // post: moves sides of rectangle outward by dx and dy
    {
	if ((-2*dx) > width) {
	    left += width/2;
	    width = 0;
	} else {
	    left -= dx;
	    width += 2*dx;
	}

	if ((-2*dy) > height) {
	    top += height/2;
	    height = 0;
	} else {
	    top -= dy;
	    height += 2*dy;
	}
    }

    /**
     * Draw a filled rectangle (in the current mode) on drawing window d.
     * @param d the target drawing window.
     * @see element.DrawingWindow#fill
     * @see element.DrawingWindow#paintMode
     * @see element.DrawingWindow#invertMode
     */
    public void fillOn(DrawingWindow d)
    // pre: d is a valid drawing window
    // post: the rectangle is filled on the drawing window d
    {
	d.fillRect(left,top,width,height);
    }

    /**
     * Erase rectangle from the drawing window d.
     *
     * @param d the target drawing window
     * @see element.DrawingWindow#clear
     */
    public void clearOn(DrawingWindow d)
    // pre: d is a valid drawing window
    // post: the rectangle is erased from the drawing window
    {
	d.clearRect(left,top,width,height);
    }

    /**
     * Draw (in the current mode) the rectangle on the drawing window
     *
     * @param d the target drawing window
     * @see element.DrawingWindow#paintMode
     * @see element.DrawingWindow#invertMode
     * @see element.DrawingWindow#draw
     */
    public void drawOn(DrawingWindow d)
    // pre: d is a valid drawing window
    // post: the rectangle is drawn on the drawing window
    {
	d.drawRect(left,top,width,height);
    }

    /**
     * Return an integer for use as a hash code
     * 
     * @return a hash code
     */
    public int hashCode()
    // post: return a hash code
    {
	return left+top+width+height;
    }

    /**
     * Return true iff this rectangle equals the other
     * 
     * @param other another valid rectangle
     * @return true if the two are equal valued, false otherwise
     */
    public boolean equals(Object other)
    // post: returns true iff two rectangles are equal
    {
	Rect that = (Rect)other;
	return (this.left == that.left) &&
	       (this.top == that.top) &&
	       (this.width == that.width) &&
	       (this.height == that.height);
    }

    /**
     * return a distinct copy of this rectangle
     * 
     * @return a distinct copy of this rectangle
     */
    public Object clone()
    // post: returns a distinct copy of the rectangle
    {
	return new Rect(this);
    }

    /**
     * Construct an AWT rectangle from this rect.
     * @return an AWT rectangle
     * @see java.awt.Rectangle
     */
    public Rectangle Rectangle()
    // post: return the equivalent AWT Rectangle
    {
	return new Rectangle(left,top,width,height);
    }

    /**
     * Construct a string representation of this rectangle
     * 
     * @return a string representation of this rectangle
     */
    public String toString()
    // post: returns a string representation of this rectangle
    {
	return "<Rectangle: left="+left+" top="+top+" width="+width+" height="+height+">";
    }
}
