// Implements a simple oval class.
// (c) 1997, 1998 duane a. bailey
package element;

/**
 * Implements an oval object.
 * An oval is the unique elipse that fits within a particular rectangle.
 * @version $Id: Oval.java,v 2.2 1999/08/05 16:18:01 bailey Exp bailey $
 * @author duane a. bailey
 */
public class Oval extends Rect
{
    /**
     * Constructs a trivial oval at the origin.
     */
    public Oval()
    // post: constructs a trivial oval at origin
    {
	this(0,0,0,0);
    }

    /**
     * Makes a copy of an existing oval.
     * @param the existing oval
     */
    public Oval(Oval r)
    // pre: r is a valid Oval
    {
	super(r);
    }

    /**
     * Construct an oval within a rectangle defined by two points.
     * @param p one bounding point
     * @param q another bounding point
     */
    public Oval(Pt p, Pt q)
    // pre: p and q are valid points
    // post: constructs an oval bounded by two points
    {
	super(p,q);
    }

    /**
     * Construct an oval bounded by an existing rectangle.
     * @param r the bounding rectangle
     */
    public Oval(Rect r)
    // post: constructs an oval bounded by a rectangle r
    {
	super(r);
    }


    /**
     * Construct an oval bounded by the rectangle with
     * upper-left (x,y), width w, and height h
     * 
     * @param x left of bounding rectangle
     * @param y top of bounding rectangle
     * @param w width of oval
     * @param h height of oval
     */
    public Oval(int x, int y, int w, int h)
    // pre:  w >= 0, h >= 0
    // post: constructs oval with top left at (x,y),
    //       width w, height h
    {
	super(x,y,w,h);
    }

    /**
     * Returns true if p is contained within oval.
     * @param p the point to be considered
     * @return true iff p is contained within the oval.
     */
    public boolean contains(Pt p)
    // pre: p is a valid point
    // post: true iff p is within the oval
    {
	// infinitely thin ovals are rects
	if ((width == 0) || (height == 0)) return super.contains(p);

	int cx = left + width/2;
	int cy = top + height/2;
	double dx = (double)(p.x()-cx);
	double dy = (double)(p.y()-cy);
	// width is radius times 2, thus the 0.25
	return ((dx/width)*(dx/width)+(dy/height)*(dy/height)) <= 0.25;
    }

    /**
     * Draw a filled oval (in the current mode) on drawing window d.
     * @param d the target drawing window.
     * @see element.DrawingWindow#fill
     * @see element.DrawingWindow#paintMode
     * @see element.DrawingWindow#invertMode
     */
    public void fillOn(DrawingWindow d)
    // pre: d is a valid drawing window
    // post: the oval is filled on the drawing window d
    {
	d.fillOval(left,top,width,height);
    }

    /**
     * Erase oval from the drawing window d.
     *
     * @param d the target drawing window
     * @see element.DrawingWindow#clear
     */
    public void clearOn(DrawingWindow d)
    // pre: d is a valid drawing window
    // post: the oval is erased from the drawing window
    {
	d.clearOval(left,top,width,height);
    }

    /**
     * Draw (in the current mode) the oval on the drawing window
     *
     * @param d the target drawing window
     * @see element.DrawingWindow#paintMode
     * @see element.DrawingWindow#invertMode
     * @see element.DrawingWindow#draw
     */
    public void drawOn(DrawingWindow d)
    // pre: d is a valid drawing window
    // post: the oval is drawn on the drawing window
    {
	d.drawOval(left,top,width,height);
    }

    /**
     * Return an integer for use as a hash code
     * 
     * @return a hash code
     */
    public int hashCode()
    // post: return a hash code
    {
	return left+top+width+height;
    }

    /**
     * Return true iff this oval equals the other
     * 
     * @param other another valid oval
     * @return true if the two are equal valued, false otherwise
     */
    public boolean equals(Object other)
    // post: returns true iff two ovals are equal
    {
	Oval that = (Oval)other;
	return (this.left == that.left) &&
	       (this.top == that.top) &&
	       (this.width == that.width) &&
	       (this.height == that.height);
    }

    /**
     * return a distinct copy of this oval
     * 
     * @return a distinct copy of this oval
     */
    public Object clone()
    // post: returns a distinct copy of the oval
    {
	return new Oval(this);
    }

    /**
     * Construct a string representation of this oval
     * 
     * @return a string representation of this oval
     */
    public String toString()
    // post: returns a string representation of this oval
    {
	return "<Oval: left="+left+" top="+top+" width="+width+" height="+height+">";
    }
}
